/* ======================================================================
 * 📌 GPT Auto Publisher Panel (v9.5 FINAL)
 * ✅ 클릭 가능한 바로가기 링크 완벽 구현
 * ✅ 도메인별 발행 현황 실시간 자동 업데이트  
 * ✅ 한글/영문 로그 컬럼 호환성 완벽 지원
 * ====================================================================== */
document.addEventListener("DOMContentLoaded", () => {
  
  function parseArray(value) {
    if (!value) return [];
    try {
      if (Array.isArray(value)) return value;
      if (typeof value === "string") {
        value = value.trim();
        if (value.startsWith("[") && value.endsWith("]")) {
          return JSON.parse(value);
        } else if (value.includes(",")) {
          return value.split(",").map(v => v.trim()).filter(v => v);
        } else {
          return [value];
        }
      }
      return [];
    } catch (e) {
      console.warn("⚠️ parseArray 변환 실패:", e.message);
      return [];
    }
  }

  // 🔑 핵심 함수: HYPERLINK 수식에서 URL 추출
  function extractUrlFromHyperlink(text) {
    if (!text) return null;
    
    // 패턴 1: =HYPERLINK("URL", "텍스트") 형태
    const hyperlinkMatch = String(text).match(/=?HYPERLINK\s*\(\s*["']([^"']+)["']/i);
    if (hyperlinkMatch && hyperlinkMatch[1]) {
      return hyperlinkMatch[1];
    }
    
    // 패턴 2: 단순 URL 형태
    const urlMatch = String(text).match(/(https?:\/\/[^\s\)]+)/i);
    if (urlMatch && urlMatch[1]) {
      return urlMatch[1];
    }
    
    return null;
  }

  const addRowBtn = document.getElementById("addRowBtn");
  const tbody = document.getElementById("postBody");
  const domainModal = document.getElementById("domainModal");
  const linkModal = document.getElementById("linkModal");

  let posts = [];
  let publishLogs = [];
  let currentPostIndex = -1;

  const SHEET_URL = "https://script.google.com/macros/s/AKfycbzEKnG9URmGjdRz773E0BC97Zi6VxIagFTzsNnj1FocTWglhPwh3C1ri69ti8JNBazD/exec";

  let isLoading = false;

  async function loadFromSheet() {
    if (isLoading) {
      console.warn("⚠️ 이미 시트 로드 중입니다. 중복 요청 차단됨.");
      return;
    }
    isLoading = true;

    if (posts.length === 0) {
      tbody.innerHTML = `<tr><td colspan="22" class="p-4 text-gray-400">시트 불러오는 중...</td></tr>`;
    }

    try {
      const callbackName = "cb_" + Date.now() + "_" + Math.floor(Math.random() * 1000);

      window[callbackName] = (data) => {
        console.log("✅ 시트 데이터 로드 성공:", data);
        
        const isEditing =
          document.querySelector("[contenteditable=true]") ||
          Array.from(document.querySelectorAll(".edit-btn")).some(btn => btn.innerText.includes("💾"));

        if (isEditing) {
          console.warn("⚠️ 수정 중... 렌더링을 건너뜁니다.");
          delete window[callbackName];
          const script = document.querySelector(`script[src*="${callbackName}"]`);
          if (script) script.remove();
          isLoading = false;
          return; 
        }

        posts = (data.rows || []).map((p) => ({
          ID: p.ID || "",
          주제: p.주제 || "",
          키워드: p.키워드 || "",
          발행도메인: parseArray(p.발행도메인),
          백링크: parseArray(p.백링크),
          발행갯수: p.발행갯수 || "",
          간격_값: p.간격_값 || 30,
          간격_단위: p.간격_단위 || "M",
          상태: p.상태 || "OFF",
          시작시간: p.시작시간 || new Date(new Date().getTime() + 9 * 60 * 60 * 1000).toISOString().slice(0, 19).replace("T", " "),
          발행결과: p.발행결과 || "대기중",
          도메인갯수: p.도메인갯수 || 0,
          백링크갯수: p.백링크갯수 || 0,
          카테고리: p.카테고리 || "",
          태그: p.태그 || "",
          랜덤모드: p.랜덤모드 === "true" || p.랜덤모드 === true,
          랜덤개수: parseInt(p.랜덤개수) || 1,
          이미지개수: parseInt(p.이미지개수) || 0,
          _JobCounter: p._JobCounter || 0,
        }));

        // 🔑 핵심 개선: 한글/영문 호환 로그 데이터 파싱
        publishLogs = (data.logs || []).map(log => {
          // 한글/영문 컬럼명 모두 지원
          const timeValue = log.Time || log["시간"] || log.time;
          const idValue = log.ID || log["ID"];
          const domainValue = log.Domain || log["도메인"];
          const topicValue = log.Topic || log["주제"];
          const resultValue = log.Result || log["결과"];
          const detailValue = log.Detail || log["상세(링크/오류)"] || log["Detail"];

          // 성공/실패 정확한 판별
          const resultText = String(resultValue || "").trim();
          const isSuccess = resultText.includes("성공") || resultText.includes("✅") || 
                           resultText.toLowerCase().includes("success");

          // URL 추출 시도
          const extractedUrl = extractUrlFromHyperlink(detailValue);

          return {
            time: timeValue ? new Date(timeValue).toLocaleString("ko-KR", { hour12: false }) : "-",
            id: idValue || "-",
            domain: domainValue || "-",
            topic: topicValue || "-",
            result: isSuccess ? "성공" : "실패",
            detail: String(detailValue || ""),
            link: extractedUrl
          };
        }).sort((a, b) => new Date(b.time) - new Date(a.time));

        renderTable();
        renderPublishLogs();
        renderDomainStats();

        delete window[callbackName];
        const script = document.querySelector(`script[src*="${callbackName}"]`);
        if (script) script.remove();
      };

      const script = document.createElement("script");
      script.src = `${SHEET_URL}?callback=${callbackName}&_=${Date.now()}`;
      script.onerror = () => {
         tbody.innerHTML = `<tr><td colspan="22" class="text-red-500 p-4">불러오기 실패: 스크립트 URL을 확인하세요.</td></tr>`;
      };
      document.body.appendChild(script);

    } catch (err) {
      console.error("❌ 시트 불러오기 실패:", err);
      tbody.innerHTML = `<tr><td colspan="22" class="text-red-500 p-4">불러오기 실패: ${err.message}</td></tr>`;
    } finally {
      isLoading = false;
    }
  }

  async function saveAllRows() {
    try {
      const iframe = document.createElement("iframe");
      iframe.name = "hiddenFrame";
      iframe.style.display = "none";
      document.body.appendChild(iframe);

      const form = document.createElement("form");
      form.method = "POST";
      form.action = SHEET_URL + "?mode=sync_sheet";
      form.target = "hiddenFrame";

      const input = document.createElement("input");
      input.type = "hidden";
      input.name = "data";
      input.value = JSON.stringify({ rows: posts });
      form.appendChild(input);
      document.body.appendChild(form);
      form.submit();
      console.log(`✅ 시트 저장 요청 전송됨 (${posts.length}개 행)`);

      iframe.onload = () => {
        console.log("📬 시트 응답 수신 (iframe)");
        setTimeout(() => {
          iframe.remove();
          form.remove();
        }, 1000);
      };

    } catch (err) {
      console.error("❌ 전체 저장 실패:", err);
    }
  }

  addRowBtn.addEventListener("click", async () => {
    const now = new Date();
    const kstTime = new Date(now.getTime() + 9 * 60 * 60 * 1000);
    const nowKstFormatted = kstTime.toISOString().slice(0, 19).replace("T", " ");

    const existingIds = posts.map(p => Number(p.ID)).filter(n => !isNaN(n));
    let newId = 1;
    while (existingIds.includes(newId)) newId++;

    const newPost = {
      ID: String(newId),
      주제: "",
      키워드: "",
      발행도메인: [],
      백링크: [],
      발행갯수: "1",
      간격_값: 30,
      간격_단위: "M",
      상태: "OFF",
      시작시간: nowKstFormatted,
      발행결과: "대기중",
      도메인갯수: 0,
      백링크갯수: 0,
      카테고리: "",
      태그: "",
      랜덤모드: false,
      랜덤개수: 1,
      이미지개수: 0,
      _JobCounter: 0,
    };

    posts.push(newPost);
    renderTable();

    const newIndex = posts.length - 1;
    const row = tbody.rows[newIndex];
    if (!row) return;

    row.querySelectorAll(".editable-cell").forEach(td => td.setAttribute("contenteditable", true));
    row.querySelector(".interval-input").disabled = false;
    row.querySelector(".interval-select").disabled = false;
    row.querySelector(".flatpickr").disabled = false;
    row.querySelector(".random-count-input").disabled = false;
    row.querySelector(".image-count-input").disabled = false;

    const btn = row.querySelector(".edit-btn");
    if (btn) btn.innerText = "💾 저장";

    await saveAllRows();
  });

  function renderTable() {
    tbody.innerHTML = "";

    if (!posts.length) {
      tbody.innerHTML = `<tr><td colspan="22" class="p-4 text-gray-400 text-center">데이터 없음</td></tr>`;
      return;
    }

    posts.sort((a, b) => Number(a.ID) - Number(b.ID));

    posts.forEach((p, i) => {
      const row = document.createElement("tr");

      const topics = Array.isArray(p.주제) ? p.주제 : parseArray(p.주제);
      const keywords = Array.isArray(p.키워드) ? p.키워드 : parseArray(p.키워드);

      let resultText = p.발행결과 || "대기중";
      if (resultText.includes("✅ 최종 발행 완료")) row.style.backgroundColor = "#e6fffa";
      else if (resultText.includes("✅")) row.style.backgroundColor = "#f0fff4";
      else if (resultText.includes("❌")) row.style.backgroundColor = "#fff5f5";

      if (p.랜덤모드 === undefined) p.랜덤모드 = false;
      if (p.랜덤개수 === undefined) p.랜덤개수 = 1;
      if (p.이미지개수 === undefined) p.이미지개수 = 0;

      row.innerHTML = `
        <td class="editable-cell" data-key="ID">${p.ID}</td>
        <td><button class="topic-btn text-blue-600" data-index="${i}">관리</button></td>
        <td>${topics.length}</td>
        <td><button class="keyword-btn text-blue-600" data-index="${i}">관리</button></td>
        <td>${keywords.length}</td>
        <td><button class="domain-btn text-blue-600" data-index="${i}">관리</button></td>
        <td>${p.도메인갯수}</td>
        <td><button class="link-btn text-blue-600" data-index="${i}">관리</button></td>
        <td>${p.백링크갯수}</td>
        <td>
          <label class="toggle">
            <input type="checkbox" data-index="${i}" class="random-mode-toggle" ${p.랜덤모드 ? "checked" : ""}>
            <span class="slider"></span>
          </label>
        </td>
        <td>
          <input type="number" min="1" value="${p.랜덤개수}" class="random-count-input w-14 text-center border rounded p-1" data-index="${i}" disabled>
        </td>
        <td>
          <input type="number" min="0" max="5" value="${p.이미지개수}" class="image-count-input w-14 text-center border rounded p-1" data-index="${i}" disabled>
        </td>
        <td class="editable-cell" data-key="발행갯수">${p.발행갯수}</td>
        <td><button class="category-btn text-blue-600" data-index="${i}">관리</button></td>
        <td>${p.카테고리 ? p.카테고리.split(",").filter(Boolean).length : 0}</td>
        <td><button class="tag-btn text-blue-600" data-index="${i}">관리</button></td>
        <td>${p.태그 ? p.태그.split(",").filter(Boolean).length : 0}</td>
        <td>
          <input type="number" value="${p.간격_값}" min="1" class="interval-input" data-index="${i}" style="width:60px;" disabled>
          <select class="interval-select" data-index="${i}" disabled>
            <option value="M" ${p.간격_단위 === "M" ? "selected" : ""}>분</option>
            <option value="H" ${p.간격_단위 === "H" ? "selected" : ""}>시</option>
            <option value="D" ${p.간격_단위 === "D" ? "selected" : ""}>일</option>
          </select>
        </td>
        <td>
          <label class="toggle">
            <input type="checkbox" ${p.상태 === "ON" ? "checked" : ""} data-index="${i}" class="status-toggle">
            <span class="slider"></span>
          </label>
        </td>
        <td><input type="text" class="flatpickr w-full text-center" value="${p.시작시간}" data-id="${p.ID}" disabled></td>
        <td>${resultText}</td>
        <td>
          <button class="edit-btn text-yellow-600" data-index="${i}">✏️ 수정</button>
          <button class="text-red-500 delete-btn" data-index="${i}">🗑 삭제</button>
        </td>
      `;

      tbody.appendChild(row);

      flatpickr(row.querySelector(".flatpickr"), {
        enableTime: true,
        dateFormat: "Y-m-d H:i:S",
        locale: "ko",
        defaultDate: p.시작시간,
        onChange: (dates, str, inst) => {
          const id = inst.element.dataset.id;
          const target = posts.find(x => x.ID === id);
          if (target) target.시작시간 = str;
        },
      });
    });

    attachEvents();
  }

  function attachEvents() {
    
    document.querySelectorAll(".edit-btn").forEach((btn) => {
      btn.onclick = async (e) => {
        const idx = e.target.dataset.index;
        const row = tbody.rows[idx];
        const post = posts[idx];

        if (btn.innerText.includes("수정")) {
          btn.innerText = "💾 저장";
          row.querySelectorAll(".editable-cell").forEach((td) => td.setAttribute("contenteditable", true));
          row.querySelector(".interval-input").disabled = false;
          row.querySelector(".interval-select").disabled = false;
          row.querySelector(".flatpickr").disabled = false;
          row.querySelector(".random-count-input").disabled = false;
          row.querySelector(".image-count-input").disabled = false;
        } else {
          row.querySelectorAll(".editable-cell").forEach((td) => {
            const key = td.dataset.key;
            post[key] = td.innerText.trim();
            td.removeAttribute("contenteditable");
          });

          post.시작시간 = row.querySelector(".flatpickr").value;
          post.간격_값 = row.querySelector(".interval-input").value;
          post.간격_단위 = row.querySelector(".interval-select").value;
          row.querySelector(".interval-input").disabled = true;
          row.querySelector(".interval-select").disabled = true;
          row.querySelector(".flatpickr").disabled = true;
          row.querySelector(".random-count-input").disabled = true;

          const imageInput = row.querySelector(".image-count-input");
          if (imageInput) {
            post.이미지개수 = parseInt(imageInput.value) || 0;
            imageInput.disabled = true;
          }

          const rounds = Number(post.발행갯수) || 1;
          const domainCount = post.발행도메인.length > 0 ? post.발행도메인.length : 1;
          post.발행갯수 = rounds * domainCount;
          post._JobCounter = 0;
          row.querySelector("[data-key='발행갯수']").innerText = post.발행갯수;

          post.발행결과 = "대기중";
          post.상태 = "OFF";
          row.cells[20].innerText = "대기중";
          row.querySelector(".status-toggle").checked = false;
          row.style.backgroundColor = "";

          btn.innerText = "✏️ 수정";
          await saveAllRows();
        }
      };
    });

    document.querySelectorAll(".status-toggle").forEach((chk) => {
      chk.onchange = async (e) => {
        const idx = e.target.dataset.index;
        const post = posts[idx];
        post.상태 = e.target.checked ? "ON" : "OFF";

        if (post.상태 === "ON") {
          post.발행결과 = "🚀 발행 요청 전송 중...";
          if (tbody.rows[idx]) tbody.rows[idx].cells[20].innerText = post.발행결과;
          await saveAllRows();

          fetch(`${SHEET_URL}?action=run_job&_=${Date.now()}`)
            .then((res) => res.text())
            .then((txt) => {
              console.log("✅ 발행 요청 완료:", txt);
              post.발행결과 = "✅ 발행 요청 완료 (GAS에서 처리 중)";
              if (tbody.rows[idx]) tbody.rows[idx].cells[20].innerText = post.발행결과;
            })
            .catch(async (err) => {
              console.error("❌ GAS 호출 실패:", err);
              post.발행결과 = "❌ GAS 호출 실패";
              if (tbody.rows[idx]) tbody.rows[idx].cells[20].innerText = post.발행결과;
              await saveAllRows();
            });
        } else {
          post.발행결과 = "⏸️ 발행 중지됨";
          if (tbody.rows[idx]) tbody.rows[idx].cells[20].innerText = post.발행결과;
          await saveAllRows();
        }
      };
    });

    document.querySelectorAll(".delete-btn").forEach((btn) => {
      btn.onclick = async (e) => {
        const idx = e.target.dataset.index;
        if (!confirm("정말 삭제하시겠습니까? (로그는 유지됩니다)")) return;
        posts.splice(idx, 1);
        renderTable();
        renderDomainStats();
        await saveAllRows();
      };
    });

    document.querySelectorAll(".domain-btn").forEach((b) => (b.onclick = openDomainModal));
    document.querySelectorAll(".link-btn").forEach((b) => (b.onclick = openLinkModal));
    document.querySelectorAll(".category-btn").forEach((b) => (b.onclick = openCategoryModal));
    document.querySelectorAll(".tag-btn").forEach((b) => (b.onclick = openTagModal));
    document.querySelectorAll(".topic-btn").forEach((b) => (b.onclick = openTopicModal));
    document.querySelectorAll(".keyword-btn").forEach((b) => (b.onclick = openKeywordModal));

    document.querySelectorAll(".random-mode-toggle").forEach(chk => {
      chk.onchange = async (e) => {
        const idx = e.target.dataset.index;
        posts[idx].랜덤모드 = e.target.checked;
        await saveAllRows();
      };
    });

    document.querySelectorAll(".random-count-input").forEach(inp => {
      inp.onchange = async (e) => {
        const idx = e.target.dataset.index;
        posts[idx].랜덤개수 = parseInt(e.target.value) || 1;
        await saveAllRows();
      };
    });
    
    document.querySelectorAll(".image-count-input").forEach(inp => {
      inp.onchange = async (e) => {
        const idx = e.target.dataset.index;
        posts[idx].이미지개수 = parseInt(e.target.value) || 0;
        await saveAllRows();
      };
    });
  }

  // 모든 모달 관리 함수들 (기존 코드 유지)
  const domainTableBody = document.querySelector("#domainModal tbody");
  const linkTableBody = document.querySelector("#linkModal tbody");
  const addDomainBtn = document.getElementById("addDomain");
  const saveDomainsBtn = document.getElementById("saveDomains");
  const closeDomainModalBtn = document.getElementById("closeDomainModal");
  const addLinkBtn = document.getElementById("addLink");
  const saveLinksBtn = document.getElementById("saveLinks");
  const closeLinkModalBtn = document.getElementById("closeLinkModal");

  function openDomainModal(e) {
    currentPostIndex = e.target.dataset.index;
    domainModal.classList.remove("hidden");
    domainTableBody.innerHTML = "";
    posts[currentPostIndex].발행도메인.forEach((url) => appendDomainRow(url));
  }

  function openLinkModal(e) {
    currentPostIndex = e.target.dataset.index;
    linkModal.classList.remove("hidden");
    linkTableBody.innerHTML = "";
    posts[currentPostIndex].백링크.forEach((l) => appendLinkRow(l));
  }

  function appendDomainRow(url = "") {
    const tr = document.createElement("tr");
    tr.innerHTML = `<td><input type="url" value="${url}" class="w-full border p-1 rounded"></td>
      <td><button class="text-red-500 delete-domain-btn">🗑</button></td>`;
    tr.querySelector(".delete-domain-btn").onclick = () => tr.remove();
    domainTableBody.appendChild(tr);
  }

  function appendLinkRow(l = { url: "", anchor: "", option: "본문", fixed: false }) {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td><input type="url" value="${l.url}" class="w-full border p-1 rounded"></td>
      <td><input type="text" value="${l.anchor}" class="w-full border p-1 rounded"></td>
      <td>
        <select class="border p-1 rounded">
          <option value="본문" ${l.option === "본문" ? "selected" : ""}>본문</option>
          <option value="랜덤" ${l.option === "랜덤" ? "selected" : ""}>랜덤</option>
          <option value="첫문단" ${l.option === "첫문단" ? "selected" : ""}>첫문단</option>
          <option value="끝문단" ${l.option === "끝문단" ? "selected" : ""}>끝문단</option>
        </select>
      </td>
      <td class="text-center">
        <input type="checkbox" class="fixed-checkbox" ${l.fixed ? "checked" : ""}>
      </td>
      <td><button class="text-red-500 delete-link-btn">🗑</button></td>
    `;
    tr.querySelector(".delete-link-btn").onclick = () => tr.remove();
    linkTableBody.appendChild(tr);
  }

  addDomainBtn.onclick = () => appendDomainRow();
  saveDomainsBtn.onclick = async () => {
    const arr = [];
    domainTableBody.querySelectorAll("input").forEach((i) => {
      if (i.value.trim()) arr.push(i.value.trim());
    });
    posts[currentPostIndex].발행도메인 = arr;
    posts[currentPostIndex].도메인갯수 = arr.length;
    await saveAllRows();
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[6].innerText = arr.length;
    domainModal.classList.add("hidden");
  };

  addLinkBtn.onclick = () => appendLinkRow();
  saveLinksBtn.onclick = async () => {
    const arr = [];
    linkTableBody.querySelectorAll("tr").forEach((tr) => {
      const inputs = tr.querySelectorAll("input");
      const select = tr.querySelector("select");
      const url = inputs[0].value.trim();
      const anchor = inputs[1].value.trim();
      const fixed = tr.querySelector(".fixed-checkbox").checked;
      if (url && anchor) arr.push({ url, anchor, option: select.value, fixed });
    });
    posts[currentPostIndex].백링크 = arr;
    posts[currentPostIndex].백링크갯수 = arr.length;
    await saveAllRows();
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[8].innerText = arr.length;
    linkModal.classList.add("hidden");
  };

  closeDomainModalBtn.onclick = () => domainModal.classList.add("hidden");
  closeLinkModalBtn.onclick = () => linkModal.classList.add("hidden");

  const previewRandomBtn = document.getElementById("previewRandomLinks");
  if (previewRandomBtn) {
    previewRandomBtn.onclick = () => {
      if (currentPostIndex === -1) return;
      const post = posts[currentPostIndex];
      const links = post.백링크 || [];
      if (links.length === 0) return alert("백링크가 없습니다.");

      const fixedLinks = links.filter(l => l.fixed);
      const nonFixed = links.filter(l => !l.fixed);
      const count = post.랜덤개수 || 1;

      const randoms = nonFixed.sort(() => 0.5 - Math.random()).slice(0, count);
      const finalSet = [...fixedLinks, ...randoms];

      alert(
        `🎯 랜덤모드: ${post.랜덤모드 ? "ON" : "OFF"}\n` +
        `🔒 고정 ${fixedLinks.length}개 + 🎲 랜덤 ${randoms.length}개 = 총 ${finalSet.length}개\n\n` +
        finalSet.map(l => `${l.anchor} (${l.url})`).join("\n")
      );
    };
  }

  // 카테고리, 태그, 주제, 키워드 모달 함수들 (기존 코드 유지)
  const categoryModal = document.getElementById("categoryModal");
  const categoryTableBody = categoryModal.querySelector("tbody");
  const addCategoryBtn = document.getElementById("addCategory");
  const saveCategoriesBtn = document.getElementById("saveCategories");
  const closeCategoryModalBtn = document.getElementById("closeCategoryModal");

  const tagModal = document.getElementById("tagModal");
  const tagTableBody = tagModal.querySelector("tbody");
  const addTagBtn = document.getElementById("addTag");
  const saveTagsBtn = document.getElementById("saveTags");
  const closeTagModalBtn = document.getElementById("closeTagModal");

  function openCategoryModal(e) {
    currentPostIndex = e.target.dataset.index;
    categoryModal.classList.remove("hidden");
    categoryTableBody.innerHTML = "";
    const cats = (posts[currentPostIndex].카테고리 || "")
      .split(",")
      .map((s) => s.trim())
      .filter(Boolean);
    cats.forEach((c) => appendCategoryRow(c));
  }

  function openTagModal(e) {
    currentPostIndex = e.target.dataset.index;
    tagModal.classList.remove("hidden");
    tagTableBody.innerHTML = "";
    const tags = (posts[currentPostIndex].태그 || "")
      .split(",")
      .map((s) => s.trim())
      .filter(Boolean);
    tags.forEach((t) => appendTagRow(t));
  }

  function appendCategoryRow(name = "") {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td><input type="text" value="${name}" class="w-full border p-1 rounded"></td>
      <td><button class="text-red-500 delete-cat-btn">🗑</button></td>
    `;
    tr.querySelector(".delete-cat-btn").onclick = () => tr.remove();
    categoryTableBody.appendChild(tr);
  }

  function appendTagRow(name = "") {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td><input type="text" value="${name}" class="w-full border p-1 rounded"></td>
      <td><button class="text-red-500 delete-tag-btn">🗑</button></td>
    `;
    tr.querySelector(".delete-tag-btn").onclick = () => tr.remove();
    tagTableBody.appendChild(tr);
  }

  addCategoryBtn.onclick = () => appendCategoryRow();
  addTagBtn.onclick = () => appendTagRow();

  saveCategoriesBtn.onclick = async () => {
    const arr = [];
    categoryTableBody.querySelectorAll("input").forEach((i) => {
      if (i.value.trim()) arr.push(i.value.trim());
    });
    posts[currentPostIndex].카테고리 = arr.join(", ");
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[14].innerText = arr.length;
    await saveAllRows();
    categoryModal.classList.add("hidden");
  };

  saveTagsBtn.onclick = async () => {
    const arr = [];
    tagTableBody.querySelectorAll("input").forEach((i) => {
      if (i.value.trim()) arr.push(i.value.trim());
    });
    posts[currentPostIndex].태그 = arr.join(", ");
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[16].innerText = arr.length;
    await saveAllRows();
    tagModal.classList.add("hidden");
  };

  closeCategoryModalBtn.onclick = () => categoryModal.classList.add("hidden");
  closeTagModalBtn.onclick = () => tagModal.classList.add("hidden");

  const topicModal = document.getElementById("topicModal");
  const topicTableBody = topicModal.querySelector("tbody");
  const addTopicBtn = document.getElementById("addTopic");
  const saveTopicsBtn = document.getElementById("saveTopics");
  const closeTopicModalBtn = document.getElementById("closeTopicModal");

  const keywordModal = document.getElementById("keywordModal");
  const keywordTableBody = keywordModal.querySelector("tbody");
  const addKeywordBtn = document.getElementById("addKeyword");
  const saveKeywordsBtn = document.getElementById("saveKeywords");
  const closeKeywordModalBtn = document.getElementById("closeKeywordModal");

  function openTopicModal(e) {
    currentPostIndex = e.target.dataset.index;
    topicModal.classList.remove("hidden");
    topicTableBody.innerHTML = "";
    const arr = parseArray(posts[currentPostIndex].주제);
    arr.forEach((v) => appendTopicRow(v));
  }

  function openKeywordModal(e) {
    currentPostIndex = e.target.dataset.index;
    keywordModal.classList.remove("hidden");
    keywordTableBody.innerHTML = "";
    const arr = parseArray(posts[currentPostIndex].키워드);
    arr.forEach((v) => appendKeywordRow(v));
  }

  function appendTopicRow(v = "") {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td><input type="text" value="${v}" class="w-full border p-1 rounded"></td>
      <td><button class="text-red-500 delete-topic-btn">🗑</button></td>
    `;
    tr.querySelector(".delete-topic-btn").onclick = () => tr.remove();
    topicTableBody.appendChild(tr);
  }

  function appendKeywordRow(v = "") {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td><input type="text" value="${v}" class="w-full border p-1 rounded"></td>
      <td><button class="text-red-500 delete-keyword-btn">🗑</button></td>
    `;
    tr.querySelector(".delete-keyword-btn").onclick = () => tr.remove();
    keywordTableBody.appendChild(tr);
  }

  addTopicBtn.onclick = () => appendTopicRow();
  addKeywordBtn.onclick = () => appendKeywordRow();

  saveTopicsBtn.onclick = async () => {
    const arr = [];
    topicTableBody.querySelectorAll("input").forEach((i) => {
      if (i.value.trim()) arr.push(i.value.trim());
    });
    posts[currentPostIndex].주제 = arr;
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[2].innerText = arr.length;
    await saveAllRows();
    topicModal.classList.add("hidden");
  };

  saveKeywordsBtn.onclick = async () => {
    const arr = [];
    keywordTableBody.querySelectorAll("input").forEach((i) => {
      if (i.value.trim()) arr.push(i.value.trim());
    });
    posts[currentPostIndex].키워드 = arr;
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[4].innerText = arr.length;
    await saveAllRows();
    keywordModal.classList.add("hidden");
  };

  closeTopicModalBtn.onclick = () => topicModal.classList.add("hidden");
  closeKeywordModalBtn.onclick = () => keywordModal.classList.add("hidden");

  function setupLogTable() {
    const existing = document.getElementById("publishLogSection");
    if (existing) return;

    const logSection = document.createElement("div");
    logSection.id = "publishLogSection";
    logSection.className = "mt-8 bg-gray-50 border rounded-lg p-4 shadow";

    logSection.innerHTML = `
      <div class="flex justify-between items-center mb-2">
        <h2 class="text-lg font-semibold">🧾 발행 로그 (10초마다 자동 갱신)</h2>
        <button id="clearLogsBtn" class="px-3 py-1 bg-red-500 text-white rounded hover:bg-red-600 text-sm">🗑 전체 삭제</button>
      </div>
      <div id="logContainer" style="max-height: 400px; overflow-y: auto;">
        <table id="logTable" class="w-full text-sm border-collapse border border-gray-300">
          <thead class="bg-gray-200">
            <tr>
              <th class="border p-2">시간</th>
              <th class="border p-2">ID</th>
              <th class="border p-2">도메인</th>
              <th class="border p-2">주제</th>
              <th class="border p-2">결과</th>
              <th class="border p-2">상세(링크/오류)</th>
            </tr>
          </thead>
          <tbody class="text-center bg-white"></tbody>
        </table>
      </div>
    `;

    document.body.appendChild(logSection);
    
    document.getElementById("clearLogsBtn").addEventListener("click", async () => {
      if (!confirm("정말 전체 로그를 삭제하시겠습니까?")) return;
      try {
        const url = `${SHEET_URL}?mode=clearLogs&_=${Date.now()}&rand=${Math.random()}`;
        const res = await fetch(url, { method: "GET", cache: "no-store" });
        const text = await res.text();
        if (!text.includes("ok")) {
          alert("⚠️ 서버 로그 삭제 실패");
          return;
        }
        publishLogs = [];
        renderPublishLogs();
        alert("🧹 모든 로그가 삭제되었습니다.");
        await loadFromSheet();
      } catch (err) {
        alert("❌ 오류: " + err.message);
      }
    });
  }

  // 🔑 핵심 개선: 클릭 가능한 바로가기 링크 렌더링
  function renderPublishLogs() {
    const tbody = document.querySelector("#logTable tbody");
    const container = document.getElementById("logContainer");
    if (!tbody || !container) return;

    const visibleLogs = publishLogs.slice(0, 60);
    tbody.innerHTML = visibleLogs.slice(0, 30).map((log) => {
      let detailHtml = log.detail || "-";
      
      // 🔑 핵심: 성공 시 클릭 가능한 바로가기 링크 생성
      if (log.result === "성공" && log.link) {
        detailHtml = `<a href="${log.link}" target="_blank" class="inline-block bg-blue-500 text-white px-3 py-1 rounded text-xs hover:bg-blue-600 transition-colors font-semibold no-underline">🔗 바로가기</a>`;
      } else if (log.result === "실패" && log.detail) {
        // 실패 시 상세 오류 메시지 툴팁 표시
        const shortDetail = log.detail.length > 40 ? log.detail.substring(0, 40) + "..." : log.detail;
        detailHtml = `<span title="${log.detail.replace(/"/g, '&quot;')}" class="text-red-600 cursor-help text-xs">${shortDetail}</span>`;
      }

      return `
        <tr class="hover:bg-gray-50">
          <td class="border p-1 text-gray-700 text-xs">${log.time}</td>
          <td class="border p-1">${log.id}</td>
          <td class="border p-1 font-medium">${log.domain}</td>
          <td class="border p-1 text-xs">${log.topic}</td>
          <td class="border p-1 font-semibold ${log.result === "성공" ? "text-green-600" : "text-red-600"}">${log.result}</td>
          <td class="border p-1">${detailHtml}</td>
        </tr>`;
    }).join("");
  }

  // 🔑 핵심 개선: 도메인별 통계 정확화
  function renderDomainStats() {
    let section = document.getElementById("domainStatsSection");
    if (!section) {
      section = document.createElement("div");
      section.id = "domainStatsSection";
      section.className = "mt-8 bg-white border rounded-lg p-4 shadow";
      document.body.appendChild(section);
    }
    
    const domainStats = {};
    publishLogs.forEach((log) => {
      if (!log.domain || log.domain === "-") return;
      if (!domainStats[log.domain]) {
        domainStats[log.domain] = { success: 0, failure: 0, total: 0 };
      }
      
      if (log.result === "성공") {
        domainStats[log.domain].success++;
      } else if (log.result === "실패") {
        domainStats[log.domain].failure++;
      }
      domainStats[log.domain].total++;
    });
    
    const rows = Object.entries(domainStats)
      .sort((a, b) => b[1].success - a[1].success)
      .map(([domain, stat]) => {
        const successRate = stat.total > 0 ? ((stat.success / stat.total) * 100).toFixed(1) : "0.0";
        return `
          <tr class="hover:bg-gray-50">
            <td class="border p-2 font-medium text-left">${domain}</td>
            <td class="border p-2 text-center text-green-600 font-bold">${stat.success}</td>
            <td class="border p-2 text-center text-red-600 font-bold">${stat.failure}</td>
            <td class="border p-2 text-center text-blue-600 font-semibold">${stat.total}</td>
            <td class="border p-2 text-center text-purple-600 font-semibold">${successRate}%</td>
          </tr>`;
      }).join("");

    section.innerHTML = `
      <h2 class="text-lg font-semibold mb-3">🌐 도메인별 발행 현황 (실시간)</h2>
      <div class="overflow-x-auto">
        <table class="w-full text-sm border-collapse border border-gray-300">
          <thead class="bg-gray-200">
            <tr>
              <th class="border p-2 text-left">도메인</th>
              <th class="border p-2">발행 성공</th>
              <th class="border p-2">발행 실패</th>
              <th class="border p-2">총 발행</th>
              <th class="border p-2">성공률</th>
            </tr>
          </thead>
          <tbody>
            ${rows || '<tr><td colspan="5" class="p-3 text-center text-gray-400">아직 데이터 없음</td></tr>'}
          </tbody>
        </table>
      </div>
    `;
  }
  
  function setupDomainModal() {
    // 기존 코드 유지
  }

  setupLogTable();
  setupDomainModal();
  loadFromSheet(); 

  setInterval(() => {
    const editing = document.querySelector("[contenteditable=true]");
    const saving = Array.from(document.querySelectorAll(".edit-btn")).some(b => b.innerText.includes("💾"));
    const now = Date.now();
    if (!window.lastLoadTime) window.lastLoadTime = 0;
    if (!editing && !saving && !isLoading && now - window.lastLoadTime > 12000) {
      window.lastLoadTime = now;
      loadFromSheet();
    }
  }, 10000);
});
